#ifndef RAT_PLUGIN_H
#define RAT_PLUGIN_H

// RAT_API_VER 1 corresponds to versions 1.3 and 1.4 of the released RatCog system.
// RAT_API_VER 2 corresponds to version 1.5 of RatCog.

/*
	CGP, Modified, 11/12/00: Created unified rat API.
		Added DatabaseMessage data member so we can evaluate the rat through adding records
		from the rat into the database.
	CGP, Modified, 12/24/00: The rats no longer have an explicit message receipt
		interface. Instead, when the experimenter puts the rat on the maze this allows
		the rat to make action and perception API calls. These calls do not specifically
		block. When the experimenter takes the rat off the maze, this blocks
		the rat from performing any actions and perceiving. The rat does not sense
		this at the present time, his call to perform an action or perception just
		block until he is put back on the maze. Moved code implementing
		API functions to RatPlugin.cpp.
*/

#define RAT_API_VER 2

#include "RatEnvMessage.h"
#include "Debug.h"
#include "PortMessage.h"
#include "DatabaseMessage.h"
#include "ProcessLock.h"

// Needed by the specific rat plugins
extern DebugServer *GLOB_debug;

class RatPlugin {
	public: // The following are internal components of the rat API
	    RatPlugin(PortMessage *ratPort, DatabaseMessage *dbMsg, RatEnvMessage *ratEnvMsg,
	    	DebugServer *bugServer);
	    virtual ~RatPlugin();
	   	// Check to see if the version level of the user plugin is same as ours.
		bool VersionOK();
		
		/////////////////////////////////////////////////////////////////////////////
		// Redeclare and override the following virtual functions to implement
		// your rat model.
		/////////////////////////////////////////////////////////////////////////////
		
		// Use the implementation for this virtual function
		// that is used the other rat plugins (e.g., StupidRat.h).
		// This is used to check for version compatibility when loading plugins in the
		// maze simulator.
		virtual int PLUGIN_API_Version() { return -1;}
		
		// This is the function containing the main loop for the rat.
		virtual long PLUGIN_Start() { return (0L); }; // Start the rat thread;

		// This function is called when the rat is taken off the maze. Only called after
		// PLUGIN_Start has been called. Called synchronously -- that is, there is not
		// concurrent execution of PLUGIN_Start and this call.
		// duration is the length of simulated time (in hours) that the rat is take off
		// the maze.
		virtual void PLUGIN_TakeOffMaze(float duration) { };

		////////////////////////////////////////////////////////////////////////////////
	 	// The following RAT_ functions make up the rat API for programming rat models.
	 	////////////////////////////////////////////////////////////////////////////////
	 	
		// wait (block) a given number of seconds
		void RAT_Wait(float secs);

	 	// For evaluation of newly coded rat models (i.e., for testing rat models).
	 	void RAT_AddDatabaseRecord(BMessage *record);
	 	int RAT_GetCurrentTrialType(); // -1 if not available from experimenter

		// Rotate rat +/- a number of degrees from his current heading.
	  	void RAT_Rotate(int degrees);
	  	
	  	// If you are in the center, check if door ahead (+/- 10 degrees of rats' heading) is open.
	  	// If you are on an arm, check if you are facing the door and it is open
	  	// (you must be within +/- 45 degrees of the open door to be considered to be facing
	  	// it).
	  	bool RAT_IsOpen();
	  	
	  	// Move in direction we are headed. Returns true iff the move succeeded.
	  	bool RAT_Move();
	  	
	  	// Check if there is food at the current location.
	  	bool RAT_FoodHere();
	  	
	  	// Eat food at current location. Returns true iff eating food succeeded.
	  	bool RAT_EatFood();
	  	
   	  	// Enable rat to look & get view of distal landmarks. Rat looks off maze facing
   	  	// in current heading and gets back a vector of cue information representing
   	  	// landmarks +/- 45 degrees (90 degree field of view) of current heading.
   	  	// Vector is owned by caller and contains LANDMARK_VIEW_WIDTH elements.
	  	float *RAT_Look();
	  	
	  	// Let the user know that something has happened. These should be used only
		// very infrequently, otherwise, it will be annoying to the user.
		// Produces a GUI message.
		void RAT_UserMessage(char *msg);
		
		// Rotate the rat until we're exactly looking at a particular view.
		// Returns true iff an exactly matching view could be found.
		// Current heading of rat is undefined if false is returned.
		bool RAT_RotateUntilView(float *view);
     
	private: // Function members *not* shared with derived classes		
		void BlockUntilRatOnMaze(); // called before perception/action function calls
		void Release(); // called after these calls.
		
	private: // data members *not* shared with derived classes
	    RatEnvMessage *envMsg; // to pass messages to environment
	    PortMessage *ratPort; // receiving messages from experimenter
	    DatabaseMessage *dbMsg; // to evaluate rat brain
	    bool ratOnMaze; // is the rat on the maze?
	    int16 trialType; // some experimenters pass this with message
	    bool stepMode; // is rat in single step mode?
};
#endif
